#!/usr/bin/python
# This script is used to plot the spectrum generated by Qbics.
#
# Please cite: J. Phys. Chem. C 2022, 126, 8720−8733
#
# In this file, the spectrum is plot using the formula:
#
# epsilon_Lrmolrcm = C * sum(strength * exp(-((e_eV-energy_eV)/sigma_eV)**2)) / sigma_eV
#
# where 
#  C is a constant.
#  strength is the oscillator strength.
#  energy_eV is the energy of the state.
#  sigma_eV is the width of the Gaussian function.
#  e_eV is the energy of the point to be calculated.
#  epsilon_Lrmolrcm is the strength of the point e_eV.

import math
import numpy as np
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec
import os

def get_spec(eL_eV, eH_eV, num_ps, energy_ev, strength, sigma_eV):
    e_eV = np.linspace(eL_eV, eH_eV, num_ps)
    epsilon_Lrmolrcm = np.zeros(e_eV.size)
    C = 1.3062974E+8 
    for i, _ in enumerate(e_eV):
        epsilon_Lrmolrcm[i] = (strength*np.exp(-((e_eV[i]-energy_eV)/sigma_eV)**2)).sum()/sigma_eV*C
    return e_eV, epsilon_Lrmolrcm

if __name__ == "__main__":
    fn = "X-spectrum.txt"       # Spectrum file name.
    eL_eV = 4                   # Lower energy limit.
    eH_eV = 13                  # Higher energy limit.
    sigma_eV = 0.2              # Sigma value.
    num_ps = 500                # Number of points.
    use_angle = False           # Whether to use angle dependence.
    incident_angles = [i*30 for i in range(4)] # Incident angles.

    # You do NOT need to modify things below.
    if use_angle:
        data = np.loadtxt(fn,  skiprows=1).transpose()
        energy_eV = data[2] 
        f = data[3]    
        dmx = data[4]
        dmy = data[5]
        dmz = data[6]
        for ti in incident_angles:
            cosei = math.cos(ti*math.pi/180.)
            cos2ei = cosei*cosei
            sin2ei = 1-cos2ei                
            f_eff = f.copy()
            for i in range(len(f)):                
                sin2et =(dmx[i]*dmx[i]+dmy[i]*dmy[i])/(dmx[i]*dmx[i]+dmy[i]*dmy[i]+dmz[i]*dmz[i])
                cos2et = 1-sin2et                
                eff = 0.5*sin2ei*sin2et+cos2ei*cos2et
                f_eff[i] = f_eff[i]*eff
            e_eV, epsilon_Lrmolrcm = get_spec(eL_eV, eH_eV, num_ps, energy_eV, f_eff, sigma_eV)
            plt.plot(e_eV, epsilon_Lrmolrcm, "-", label = str(ti))
            plt.legend()              
    else:
        data = np.loadtxt(fn, skiprows=1).transpose()
        energy_eV = data[2] 
        f = data[3]    
        e_eV, epsilon_Lrmolrcm = get_spec(eL_eV, eH_eV, num_ps, energy_eV, f, sigma_eV)
        plt.plot(e_eV, epsilon_Lrmolrcm, "-")    
    plt.xlabel("Energy (eV)")
    plt.ylabel("Strength (L/mol/cm)")
    plt.show()